<?php
// Enable error reporting
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);
ini_set('log_errors', 1);
ini_set('error_log', __DIR__ . '/errors.log');

// Start session
session_start();

// Checkpoint 1: Basic script execution
file_put_contents(__DIR__ . '/debug.log', "Checkpoint 1: Script started\n", FILE_APPEND);

// Check if config.php exists
if (!file_exists(__DIR__ . '/config.php')) {
    file_put_contents(__DIR__ . '/debug.log', "Error: config.php not found\n", FILE_APPEND);
    die('config.php not found. Please check your configuration.');
}
include __DIR__ . '/config.php';

// Checkpoint 2: Config included
file_put_contents(__DIR__ . '/debug.log', "Checkpoint 2: config.php included\n", FILE_APPEND);

// Verify database connection
if (!isset($conn) || $conn->connect_error) {
    file_put_contents(__DIR__ . '/debug.log', "Error: Database connection failed: " . ($conn->connect_error ?? 'No connection object') . "\n", FILE_APPEND);
    die('Database connection failed. Please check config.php.');
}

// Checkpoint 3: Database connected
file_put_contents(__DIR__ . '/debug.log', "Checkpoint 3: Database connected\n", FILE_APPEND);

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Checkpoint 4: POST request received
    file_put_contents(__DIR__ . '/debug.log', "Checkpoint 4: POST request received\n", FILE_APPEND);

    $email = filter_var($_POST['email'] ?? '', FILTER_SANITIZE_EMAIL);

    if (filter_var($email, FILTER_VALIDATE_EMAIL)) {
        // Checkpoint 5: Valid email
        file_put_contents(__DIR__ . '/debug.log', "Checkpoint 5: Valid email: $email\n", FILE_APPEND);

        // Check if email exists
        $stmt = $conn->prepare('SELECT id FROM users WHERE email = ?');
        if (!$stmt) {
            file_put_contents(__DIR__ . '/debug.log', "Error: Database prepare error: " . $conn->error . "\n", FILE_APPEND);
            die('Internal server error. Please try again later.');
        }
        $stmt->bind_param('s', $email);
        $stmt->execute();
        $result = $stmt->get_result();
        $user = $result->fetch_assoc();
        $stmt->close();

        // Checkpoint 6: User query executed
        file_put_contents(__DIR__ . '/debug.log', "Checkpoint 6: User query executed, user " . ($user ? 'found' : 'not found') . "\n", FILE_APPEND);

        // Generic success message to prevent email enumeration
        $success_message = "If this email exists, a recovery link will be sent. <a href='index.php'>Back to Login</a>";

        if ($user) {
            try {
                // Generate and store reset token
                $token = bin2hex(random_bytes(32));
                $expires = date('Y-m-d H:i:s', strtotime('+1 hour'));

                $stmt = $conn->prepare('INSERT INTO password_resets (email, token, expires_at) VALUES (?, ?, ?) 
                                      ON DUPLICATE KEY UPDATE token = ?, expires_at = ?');
                if (!$stmt) {
                    file_put_contents(__DIR__ . '/debug.log', "Error: Database prepare error: " . $conn->error . "\n", FILE_APPEND);
                    die('Internal server error. Please try again later.');
                }
                $stmt->bind_param('sssss', $email, $token, $expires, $token, $expires);
                $stmt->execute();
                $stmt->close();

                // Checkpoint 7: Token stored
                file_put_contents(__DIR__ . '/debug.log', "Checkpoint 7: Token stored for email: $email\n", FILE_APPEND);

                // Prepare email
                $reset_link = "https://sveniverse.com/reset_password.php?token=" . urlencode($token);
                $subject = 'Password Reset Request';
                $body = "Hi,\r\n\r\n"
                      . "We received a password reset request for your account.\r\n\r\n"
                      . "Click the link below to reset your password:\r\n\r\n"
                      . "$reset_link\r\n\r\n"
                      . "This link will expire in 1 hour.\r\n\r\n"
                      . "If you did not request this, please ignore this email.";
                $headers = "From: Sveniverse Support <no-reply@sveniverse.com>\r\n"
                         . "Reply-To: no-reply@sveniverse.com\r\n"
                         . "X-Mailer: PHP/" . phpversion() . "\r\n"
                         . "MIME-Version: 1.0\r\n"
                         . "Content-Type: text/plain; charset=UTF-8";

                // Send email using mail()
                if (mail($email, $subject, $body, $headers)) {
                    file_put_contents(__DIR__ . '/debug.log', "Checkpoint 8: Email sent successfully to $email\n", FILE_APPEND);
                    echo $success_message;
                } else {
                    file_put_contents(__DIR__ . '/debug.log', "Error: Failed to send email to $email\n", FILE_APPEND);
                    echo $success_message; // Don't reveal email failure to user
                }
            } catch (Exception $e) {
                file_put_contents(__DIR__ . '/debug.log', "Error: Exception - " . $e->getMessage() . "\n", FILE_APPEND);
                echo $success_message;
            }
        } else {
            file_put_contents(__DIR__ . '/debug.log', "Checkpoint 8: No user found for $email, showing success message\n", FILE_APPEND);
            echo $success_message;
        }
    } else {
        file_put_contents(__DIR__ . '/debug.log', "Checkpoint 5: Invalid email\n", FILE_APPEND);
        echo "Invalid email address. <a href='recover.php'>Try again</a>";
    }
} else {
    file_put_contents(__DIR__ . '/debug.log', "Checkpoint 4: Not a POST request, redirecting\n", FILE_APPEND);
    header("Location: recover.php");
    exit;
}
?>