<?php
session_start();
include 'config.php'; // $conn is available

// Step 1: Check if token is in URL
if (!isset($_GET['token'])) {
    echo "Invalid request.";
    exit;
}

$token = $_GET['token'];

// Step 2: Look up the token in the database
$stmt = $conn->prepare('SELECT email, expires_at FROM password_resets WHERE token = ?');
$stmt->bind_param('s', $token);
$stmt->execute();
$result = $stmt->get_result();
$resetRequest = $result->fetch_assoc();
$stmt->close();

// Step 3: Validate token
if (!$resetRequest) {
    echo "Invalid or expired token.";
    exit;
}

if (strtotime($resetRequest['expires_at']) < time()) {
    echo "Token has expired. Please request a new password reset.";
    exit;
}

// Step 4: Handle new password form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $new_password = $_POST['new_password'];
    $confirm_password = $_POST['confirm_password'];

    if (empty($new_password) || empty($confirm_password)) {
        echo "Please fill in all fields.";
    } elseif ($new_password !== $confirm_password) {
        echo "Passwords do not match.";
    } elseif (strlen($new_password) < 6) {
        echo "Password must be at least 6 characters.";
    } else {
        // Password is valid, hash it
        $hashed_password = password_hash($new_password, PASSWORD_DEFAULT);

        // Update the user's password
        $stmt = $conn->prepare('UPDATE users SET password = ? WHERE email = ?');
        $stmt->bind_param('ss', $hashed_password, $resetRequest['email']);
        $stmt->execute();
        $stmt->close();

        // Delete the password reset request
        $stmt = $conn->prepare('DELETE FROM password_resets WHERE email = ?');
        $stmt->bind_param('s', $resetRequest['email']);
        $stmt->execute();
        $stmt->close();

        echo "Password has been reset successfully! <a href='index.php'>Login here</a>";
        exit;
    }
}
?>

<!-- HTML Form for new password -->
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Reset Password</title>
</head>
<body>
    <h2>Reset Your Password</h2>
    <form method="post">
        <label>New Password:</label><br>
        <input type="password" name="new_password" required><br><br>

        <label>Confirm New Password:</label><br>
        <input type="password" name="confirm_password" required><br><br>

        <button type="submit">Reset Password</button>
    </form>
</body>
</html>
